<?xml version="1.0" encoding="UTF-8"?>

<!-- 
	File   : TxResult.xsl
	brief  : 通信結果レポート用XSLファイル
	author :.
	date   : 2011/XX/XX		新規作成.
	
-->

<!-- スタイルシートタグ -->
<xsl:stylesheet version="1.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform">

	<!-- 出力方法の指定 -->
	<xsl:output method="html" encoding="UTF-8"/>

	<!--    brief: メイン制御部
		param: なし	 -->
	<xsl:template match="Report">

		<html>
		<!-- CSSデータ一覧 -->
		<xsl:element name="style">
			<xsl:attribute name="type">text/css</xsl:attribute>
			<xsl:comment>
			div.PAGEEND { page-break-after: always; }
			body { font-size: 9pt }
			table { font-size: 9pt }
			</xsl:comment>
		</xsl:element>

		<body>

		<!-- 1ページ最大リスト件数 -->
		<xsl:variable name="ListMaxNum"  select="20"/>

		<!-- 最終ページ数算出 -->
		<xsl:variable name="ListNum" select="count(/Report/List)" />
		<xsl:variable name="MaxPageNum">
			<xsl:choose>
				<xsl:when test="$ListNum = 0">
					<xsl:value-of name="MaxPageNum" select="1"/>
				</xsl:when>
				<xsl:when test="$ListNum mod $ListMaxNum = 0">
					<xsl:value-of name="MaxPageNum" select="floor($ListNum div $ListMaxNum)"/>
				</xsl:when>
				<xsl:otherwise>
					<xsl:value-of name="MaxPageNum" select="floor($ListNum div $ListMaxNum + 1)"/>
				</xsl:otherwise>
			</xsl:choose>
		</xsl:variable>

		<!-- 仕向け -->
		<xsl:variable name="MarketingArea" select="/Report/Header/MarketingArea" />

		<!-- 対応言語 -->
		<xsl:variable name="Language" select="/Report/Header/Language" />

		<!-- レポート作成実行 -->
		<xsl:variable name="PageCount" select="1" />
		<xsl:call-template name="MakePage">
			<xsl:with-param name="MarketingArea" select="$MarketingArea"/>
			<xsl:with-param name="Language" select="$Language"/>
			<xsl:with-param name="PageCount" select="$PageCount"/>
			<xsl:with-param name="MaxPageNum" select="$MaxPageNum"/>
			<xsl:with-param name="ListMaxNum" select="$ListMaxNum"/>
		</xsl:call-template>

		</body>
		</html>
	</xsl:template><!-- xsl:template match="Report" -->

	<!--    brief: レポート作成
		param: 仕向け設定
		　　　 言語設定
		　　　 現状ページ数
		　　　 印字最大ページ数
		　　　 1ページ最大リスト件数　　　　-->
	<xsl:template name="MakePage">
		<!-- パラメータ -->
		<xsl:param name="MarketingArea"/>
		<xsl:param name="Language" />
		<xsl:param name="PageCount"/>
		<xsl:param name="MaxPageNum"/>
		<xsl:param name="ListMaxNum"/>

		<!-- ページヘッダ作成 -->
		<xsl:call-template name="MakePageHeader">
			<xsl:with-param name="Language" select="$Language"/>
			<xsl:with-param name="MarketingArea" select="$MarketingArea"/>
			<xsl:with-param name="LocalPageNum" select="$PageCount"/>
		</xsl:call-template>

		<!-- レポートフォーマット表作成  -->
		<xsl:call-template name="MakeReportMatrix">
			<xsl:with-param name="Language" select="$Language"/>
			<xsl:with-param name="LocalPageNum" select="$PageCount"/>
			<xsl:with-param name="ListMaxNum" select="$ListMaxNum"/>
		</xsl:call-template>

		<!-- 改ページ挿入 -->
		<xsl:if test="$PageCount != $MaxPageNum">
			<xsl:element name="div">
				<xsl:attribute name="class">PAGEEND</xsl:attribute>
			</xsl:element>
		</xsl:if>

		<!-- 最終印字ページまで未達成の間、再帰呼び出し -->
		<xsl:if test="$PageCount &lt; $MaxPageNum">
			<xsl:call-template name="MakePage">
				<xsl:with-param name="MarketingArea" select="$MarketingArea"/>
				<xsl:with-param name="Language" select="$Language"/>
				<xsl:with-param name="PageCount" select="$PageCount+1"/>
				<xsl:with-param name="MaxPageNum" select="$MaxPageNum"/>
				<xsl:with-param name="ListMaxNum" select="$ListMaxNum"/>
			</xsl:call-template>
		</xsl:if>
	</xsl:template><!-- "MakePage" -->

	<!--    brief: ページヘッダ作成
		param: 仕向け設定
		　　　 現状ページ数　　　　-->
	<xsl:template name="MakePageHeader">
		<!-- パラメータ -->
		<xsl:param name="Language"/>
		<xsl:param name="MarketingArea"/>
		<xsl:param name="LocalPageNum"/>

		<xsl:variable name="Year" select="Header/Year"/>
		<xsl:variable name="Month" select="Header/Month"/>
		<xsl:variable name="Day" select="Header/Day"/>
		<xsl:variable name="Hour" select="Header/Hour"/>
		<xsl:variable name="Minute" select="Header/Minute"/>
		<!-- XSLで使用する変数への代入 ここまで  -->

		<!-- レポートタイトル設定 -->
		<xsl:element name="div">
			<xsl:attribute name="align">center</xsl:attribute>
			<xsl:call-template name="ChangeLangStr">
				<xsl:with-param name="Language" select="$Language"/>
				<xsl:with-param name="TitleName" select="'ReportTitle'"/>
			</xsl:call-template>
		</xsl:element>

		<!-- ヘッダデータ整形 -->
		<xsl:element name="div">
			<xsl:attribute name="align">right</xsl:attribute>

			<!-- ページ番号 -->
			<xsl:value-of select="concat('P ' ,$LocalPageNum)"/><br/>

			<!-- 日付/時刻 -->
			<xsl:choose>
				<xsl:when test="$MarketingArea = 1"><!-- 米国向け [Month/Day/Year Hour:Minute] -->
					<xsl:value-of select="format-number($Month, '00')"/>/<xsl:value-of select="format-number($Day, '00')"/>/<xsl:value-of select="format-number($Year, '0000')"/>
					<xsl:text disable-output-escaping="yes">&amp;nbsp;</xsl:text>
					<xsl:value-of select="format-number($Hour, '00')"/>:<xsl:value-of select="format-number($Minute, '00')"/><br/>
				</xsl:when>
				<xsl:when test="$MarketingArea = 2"><!-- 欧州向け [Day/Month/Year Hour:Minute] -->
					<xsl:value-of select="format-number($Day, '00')"/>/<xsl:value-of select="format-number($Month, '00')"/>/<xsl:value-of select="format-number($Year, '0000')"/>
					<xsl:text disable-output-escaping="yes">&amp;nbsp;</xsl:text>
					<xsl:value-of select="format-number($Hour, '00')"/>:<xsl:value-of select="format-number($Minute, '00')"/><br/>
				</xsl:when>
				<xsl:otherwise><!-- 日本向け/default [Year/Month/Day Hour:Minute] -->
					<xsl:value-of select="format-number($Year, '0000')"/>/<xsl:value-of select="format-number($Month, '00')"/>/<xsl:value-of select="format-number($Day, '00')"/>
					<xsl:text disable-output-escaping="yes">&amp;nbsp;</xsl:text>
					<xsl:value-of select="format-number($Hour, '00')"/>:<xsl:value-of select="format-number($Minute, '00')"/><br/>
				</xsl:otherwise>
			</xsl:choose>

			<!-- シリアルナンバー -->
			 <xsl:value-of select="concat('Serial No. ' ,Header/SerialNo)"/><br/>

			<!-- トータルカウンタ -->
			<xsl:value-of select="concat('TC: ' ,Header/TotalCounter)"/><br/><br/>

		</xsl:element>
	</xsl:template><!-- "MakePageHeader" -->

	<!--    brief: レポートフォーマット表作成
		param: 対応言語設定
		       現状ページ数
		　　　 1ページ最大リスト件数　　　　-->
	<xsl:template name="MakeReportMatrix">
		<!-- パラメータ -->
		<xsl:param name="Language"/>
		<xsl:param name="LocalPageNum"/>
		<xsl:param name="ListMaxNum"/>

		<!-- テーブルヘッダ部作成 -->
		<xsl:element name="table">
		<xsl:attribute name="border">1</xsl:attribute>
		<xsl:attribute name="cellspacing">0</xsl:attribute>
		<xsl:attribute name="bordercolor">black</xsl:attribute>
		<xsl:attribute name="width">100%</xsl:attribute>
			<xsl:element name="tr">
				<!-- 相手先 -->
				<xsl:element name="td">
					<xsl:attribute name="align">center</xsl:attribute>
					<xsl:attribute name="valign">top</xsl:attribute>
					<xsl:attribute name="width">22%</xsl:attribute>
					<xsl:call-template name="ChangeLangStr">
						<xsl:with-param name="Language" select="$Language"/>
						<xsl:with-param name="TitleName" select="'RemoteIDTitle'"/>
					</xsl:call-template>
				</xsl:element>
				<!-- 開始時刻 -->
				<xsl:element name="td">
					<xsl:attribute name="align">center</xsl:attribute>
					<xsl:attribute name="valign">top</xsl:attribute>
					<xsl:attribute name="width">12%</xsl:attribute>
					<xsl:call-template name="ChangeLangStr">
						<xsl:with-param name="Language" select="$Language"/>
						<xsl:with-param name="TitleName" select="'StartTimeTitle'"/>
					</xsl:call-template>
				</xsl:element>
				<!-- 時間 -->
				<xsl:element name="td">
					<xsl:attribute name="align">center</xsl:attribute>
					<xsl:attribute name="valign">top</xsl:attribute>
					<xsl:attribute name="width">10%</xsl:attribute>
					<xsl:call-template name="ChangeLangStr">
						<xsl:with-param name="Language" select="$Language"/>
						<xsl:with-param name="TitleName" select="'DurationTimeTitle'"/>
					</xsl:call-template>
				</xsl:element>
				<!-- 枚数 -->
				<xsl:element name="td">
					<xsl:attribute name="align">center</xsl:attribute>
					<xsl:attribute name="valign">top</xsl:attribute>
					<xsl:attribute name="width">8%</xsl:attribute>
					<xsl:call-template name="ChangeLangStr">
						<xsl:with-param name="Language" select="$Language"/>
						<xsl:with-param name="TitleName" select="'PageNoTitle'"/>
					</xsl:call-template>
				</xsl:element>
				<!-- 結果 -->
				<xsl:element name="td">
					<xsl:attribute name="align">center</xsl:attribute>
					<xsl:attribute name="valign">top</xsl:attribute>
					<xsl:attribute name="width">8%</xsl:attribute>
					<xsl:call-template name="ChangeLangStr">
						<xsl:with-param name="Language" select="$Language"/>
						<xsl:with-param name="TitleName" select="'ResultTitle'"/>
					</xsl:call-template>
				</xsl:element>
				<!-- 備考 -->
				<xsl:element name="td">
					<xsl:attribute name="align">center</xsl:attribute>
					<xsl:attribute name="valign">top</xsl:attribute>
					<xsl:attribute name="width">40%</xsl:attribute>
					<xsl:call-template name="ChangeLangStr">
						<xsl:with-param name="Language" select="$Language"/>
						<xsl:with-param name="TitleName" select="'NotesTitle'"/>
					</xsl:call-template>
				</xsl:element>
			</xsl:element><!-- "tr" -->

			<!-- テーブルデータ部作成 -->
			<xsl:apply-templates select="List">
				<xsl:with-param name="Language" select="$Language"/>
				<xsl:with-param name="MAXRecordNum" select="$ListMaxNum*$LocalPageNum"/>
				<xsl:with-param name="MINRecordNum" select="$ListMaxNum*($LocalPageNum -1)"/>
			</xsl:apply-templates>

		</xsl:element><!-- "table" -->

		<!-- レポートフォーマット補足説明部作成 -->
		<xsl:call-template name="Explanation">
				<xsl:with-param name="Language" select="$Language"/>
		</xsl:call-template>

	</xsl:template><!-- "MakeReportMatrix" -->

	<!--    brief:  テーブルデータ部作成
		param:  対応言語設定
			最大レコード数
			最小レコード数　　　　-->
	<xsl:template match="List">
		<!-- パラメータ -->
		<xsl:param name="Language"/>
		<xsl:param name="MAXRecordNum"/>
		<xsl:param name="MINRecordNum"/>
		<xsl:if test="$MINRecordNum &lt; @DataNum">
			<xsl:if test="@DataNum &lt;= $MAXRecordNum">
			<xsl:element name="tr">
				<!-- 相手先（データ） -->
				<xsl:element name="td">
					<xsl:attribute name="align">left</xsl:attribute>
					<xsl:attribute name="valign">top</xsl:attribute>
					<xsl:attribute name="width">22%</xsl:attribute>
					<xsl:value-of select="RemoteID"/>
				</xsl:element>
				<!-- 開始時刻（データ） -->
				<xsl:element name="td">
					<xsl:attribute name="align">center</xsl:attribute>
					<xsl:attribute name="valign">top</xsl:attribute>
					<xsl:attribute name="width">12%</xsl:attribute>
					<xsl:value-of select="concat(format-number(StartMonth, '00'), '/')"/>
					<xsl:value-of select="format-number(StartDay, '00')"/>
					<xsl:text disable-output-escaping="yes">&amp;nbsp;</xsl:text>
					<xsl:value-of select="concat(format-number(StartHour, '00'), ':')"/>
					<xsl:value-of select="format-number(StartMin, '00')"/>
				</xsl:element>
				<!-- 時間（データ） -->
				<xsl:element name="td">
					<xsl:attribute name="align">center</xsl:attribute>
					<xsl:attribute name="valign">top</xsl:attribute>
					<xsl:attribute name="width">10%</xsl:attribute>
					<xsl:value-of select="concat(format-number(DurationHour, '00'), ':')"/>
					<xsl:value-of select="concat(format-number(DurationMin, '00'), ':')"/>
					<xsl:value-of select="format-number(DurationSec, '00')"/>
				</xsl:element>
				<!-- 枚数（データ） -->
				<xsl:element name="td">
					<xsl:attribute name="align">center</xsl:attribute>
					<xsl:attribute name="valign">top</xsl:attribute>
					<xsl:attribute name="width">8%</xsl:attribute>
					<xsl:value-of select="concat(format-number(CommPageNo, '000'), '/')"/>
					<xsl:value-of select="format-number(DocPageNo, '000')"/>
				</xsl:element>
				<!-- 結果（データ） -->
				<xsl:element name="td">
					<xsl:attribute name="align">center</xsl:attribute>
					<xsl:attribute name="valign">top</xsl:attribute>
					<xsl:attribute name="width">8%</xsl:attribute>
					<xsl:call-template name="MakeCommResultStr">
						<xsl:with-param name="Language" select="$Language"/>
					</xsl:call-template>
				</xsl:element>
				<!-- 備考（データ） -->
				<xsl:element name="td">
					<xsl:attribute name="align">left</xsl:attribute>
					<xsl:attribute name="valign">top</xsl:attribute>
					<xsl:attribute name="width">40%</xsl:attribute>
					<xsl:call-template name="MakeNotesStr">
						<xsl:with-param name="DurationHour" select="DurationHour"/>
						<xsl:with-param name="DurationMin" select="DurationMin"/>
						<xsl:with-param name="DurationSec" select="DurationSec"/>
					</xsl:call-template>
				</xsl:element>
			</xsl:element><!-- "tr" -->
			</xsl:if>
		</xsl:if>
	</xsl:template>

	<!-- レポートフォーマット補足説明部 -->
	<xsl:template name="Explanation">
		<!-- パラメータ -->
		<xsl:param name="Language"/>

		<xsl:element name="table">
		<xsl:attribute name="border">0</xsl:attribute>
		<xsl:attribute name="width">100%</xsl:attribute>
			<xsl:element name="tr">
				<!-- 備考 -->
				<xsl:element name="td">
					<xsl:attribute name="align">left</xsl:attribute>
					<xsl:attribute name="valign">top</xsl:attribute>
					<xsl:attribute name="width">10%</xsl:attribute>
					<xsl:call-template name="ChangeLangStr">
						<xsl:with-param name="Language" select="$Language"/>
						<xsl:with-param name="TitleName" select="'NotesTitle'"/>
					</xsl:call-template>
				</xsl:element>
				<!-- 備考/説明 -->
				<xsl:element name="td">
					<xsl:attribute name="align">left</xsl:attribute>
					<xsl:attribute name="valign">top</xsl:attribute>
					<xsl:attribute name="width">90%</xsl:attribute>
					<xsl:call-template name="ChangeLangStr">
						<xsl:with-param name="Language" select="$Language"/>
						<xsl:with-param name="TitleName" select="'NotesExplanation'"/>
					</xsl:call-template>
				</xsl:element>
			</xsl:element><!-- "tr" -->
		</xsl:element><!-- "table" --><br/>

		<xsl:element name="table">
		<xsl:attribute name="border">0</xsl:attribute>
		<xsl:attribute name="width">100%</xsl:attribute>
			<xsl:element name="tr">
				<!-- 結果 -->
				<xsl:element name="td">
					<xsl:attribute name="align">left</xsl:attribute>
					<xsl:attribute name="valign">top</xsl:attribute>
					<xsl:attribute name="width">10%</xsl:attribute>
					<xsl:call-template name="ChangeLangStr">
						<xsl:with-param name="Language" select="$Language"/>
						<xsl:with-param name="TitleName" select="'ResultTitle'"/>
					</xsl:call-template>
				</xsl:element>
				<!-- 結果/説明 -->
				<xsl:element name="td">
					<xsl:attribute name="align">left</xsl:attribute>
					<xsl:attribute name="valign">top</xsl:attribute>
					<xsl:attribute name="width">90%</xsl:attribute>
					<xsl:call-template name="ChangeLangStr">
						<xsl:with-param name="Language" select="$Language"/>
						<xsl:with-param name="TitleName" select="'ResultExplanation'"/>
					</xsl:call-template>
				</xsl:element>
			</xsl:element><!-- "tr" -->
		</xsl:element><!-- "table" -->
	</xsl:template>

	<!--     brief: 通信結果文字列設定
		 param: 対応言語設定
			結果データ　-->
	<xsl:template name="MakeCommResultStr">
		<!-- パラメータ -->
		<xsl:param name="Language"/>

		<xsl:variable name="DiagnosticCodeEnable" select="DiagnosticCodeEnable"/>
		<xsl:variable name="Result1" select="Result1"/>
		<xsl:variable name="Result2" select="Result2"/>
		<xsl:variable name="Result3" select="Result3"/>

		<xsl:choose>
		<xsl:when test="$DiagnosticCodeEnable = 1">

			<xsl:variable name="ErrorWork1" select="$Result3 mod 2"/>
			<xsl:variable name="ErrorWork2" select="(floor($Result3 div 2)) mod 2"/>
			<xsl:variable name="ErrorWork3" select="(floor($Result3 div 4)) mod 2"/>
			<xsl:variable name="ErrorWork4" select="(floor($Result3 div 8)) mod 2"/>
			<xsl:variable name="ErrorWork" select="(floor($Result3 div 8)) * 10 + $ErrorWork1 + $ErrorWork2 * 2 + $ErrorWork3 * 4 + $ErrorWork4 * 8"/>

			<xsl:if test="$Result2 = 'T' or $Result2 = 'D'">
				<xsl:value-of select="'T-'"/>
				<xsl:value-of select="format-number(string($ErrorWork), '00')"/>
			</xsl:if>
			<xsl:if test="$Result2 = 'R'">
				<xsl:value-of select="'R-'"/>
				<xsl:value-of select="format-number(string($ErrorWork), '00')"/>
			</xsl:if>
			<xsl:if test="$Result2 = 'F'">
				<xsl:value-of select="'B0-'"/>
				<xsl:value-of select="format-number(string($ErrorWork), '00')"/>
			</xsl:if>
			<xsl:if test="$Result2 = 'E'">
				<xsl:value-of select="'B1-'"/>
				<xsl:value-of select="format-number(string($ErrorWork), '00')"/>
			</xsl:if>
			<xsl:if test="$Result2 = 'N'">
				<xsl:value-of select="'N-'"/>
				<xsl:value-of select="format-number(string($ErrorWork), '00')"/>
			</xsl:if>
		</xsl:when>
		<xsl:otherwise>
			<xsl:choose>
			<!-- OK -->
			<xsl:when test="$Result1 = 0">
				<xsl:call-template name="ChangeLangStr">
					<xsl:with-param name="Language" select="$Language"/>
					<xsl:with-param name="TitleName" select="'ResultOK'"/>
				</xsl:call-template>
			</xsl:when>
			<!-- S-OK -->
			<xsl:when test="$Result1 = 1">
				<xsl:call-template name="ChangeLangStr">
					<xsl:with-param name="Language" select="$Language"/>
					<xsl:with-param name="TitleName" select="'ResultS-OK'"/>
				</xsl:call-template>
			</xsl:when>
			<!-- PW-OFF -->
			<xsl:when test="$Result1 = 2">
				<xsl:call-template name="ChangeLangStr">
					<xsl:with-param name="Language" select="$Language"/>
					<xsl:with-param name="TitleName" select="'ResultPW-OFF'"/>
				</xsl:call-template>
			</xsl:when>
			<!-- TEL(R00) -->
			<xsl:when test="$Result1 = 3">
				<xsl:call-template name="ChangeLangStr">
					<xsl:with-param name="Language" select="$Language"/>
					<xsl:with-param name="TitleName" select="'ResultTEL'"/>
				</xsl:call-template>
			</xsl:when>
			<!-- NG(Txx/Rxx) -->
			<xsl:when test="$Result1 = 4">
				<xsl:call-template name="ChangeLangStr">
					<xsl:with-param name="Language" select="$Language"/>
					<xsl:with-param name="TitleName" select="'ResultNG'"/>
				</xsl:call-template>
			</xsl:when>
			<!-- 継続(Txx) -->
			<xsl:when test="$Result1 = 5">
				<xsl:call-template name="ChangeLangStr">
					<xsl:with-param name="Language" select="$Language"/>
					<xsl:with-param name="TitleName" select="'ResultCont'"/>
				</xsl:call-template>
			</xsl:when>
			<!-- 無応答(T82) -->
			<xsl:when test="$Result1 = 6">
				<xsl:call-template name="ChangeLangStr">
					<xsl:with-param name="Language" select="$Language"/>
					<xsl:with-param name="TitleName" select="'ResultNoAns'"/>
				</xsl:call-template>
			</xsl:when>
			<!-- 着信拒否(R20) -->
			<xsl:when test="$Result1 = 7">
				<xsl:call-template name="ChangeLangStr">
					<xsl:with-param name="Language" select="$Language"/>
					<xsl:with-param name="TitleName" select="'ResultRefuse'"/>
				</xsl:call-template>
			</xsl:when>
			<!-- 話中(T83) -->
			<xsl:when test="$Result1 = 8">
				<xsl:call-template name="ChangeLangStr">
					<xsl:with-param name="Language" select="$Language"/>
					<xsl:with-param name="TitleName" select="'ResultBusy'"/>
				</xsl:call-template>
			</xsl:when>
			<!-- メモリフル(R18) -->
			<xsl:when test="$Result1 = 9">
				<xsl:call-template name="ChangeLangStr">
					<xsl:with-param name="Language" select="$Language"/>
					<xsl:with-param name="TitleName" select="'ResultM-Full'"/>
				</xsl:call-template>
			</xsl:when>
			<!-- 回復不能エラー -->
			<xsl:when test="$Result1 = 10">
				<xsl:call-template name="ChangeLangStr">
					<xsl:with-param name="Language" select="$Language"/>
					<xsl:with-param name="TitleName" select="'ResultNG'"/>
				</xsl:call-template>
			</xsl:when>
			<!-- スキャナーJAM発生エラー -->
			<xsl:when test="$Result1 = 11">
				<xsl:call-template name="ChangeLangStr">
					<xsl:with-param name="Language" select="$Language"/>
					<xsl:with-param name="TitleName" select="'ResultScanJamErr'"/>
				</xsl:call-template>
			</xsl:when>
			<!-- スキャナーメモリフル -->
			<xsl:when test="$Result1 = 12">
				<xsl:call-template name="ChangeLangStr">
					<xsl:with-param name="Language" select="$Language"/>
					<xsl:with-param name="TitleName" select="'ResultM-Full'"/>
				</xsl:call-template>
			</xsl:when>
			<!-- スキャナーその他エラー -->
			<xsl:when test="$Result1 = 13">
				<xsl:call-template name="ChangeLangStr">
					<xsl:with-param name="Language" select="$Language"/>
					<xsl:with-param name="TitleName" select="'ResultS-OK'"/>
				</xsl:call-template>
			</xsl:when>
			<!-- 強制メモリ受信ボックスからの印刷(PRINT) -->
			<xsl:when test="$Result1 = 14">
				<xsl:call-template name="ChangeLangStr">
					<xsl:with-param name="Language" select="$Language"/>
					<xsl:with-param name="TitleName" select="'ResultPrint'"/>
				</xsl:call-template>
			</xsl:when>
			<!-- 強制メモリ受信ボックスからの削除(DEL) -->
			<xsl:when test="$Result1 = 15">
				<xsl:call-template name="ChangeLangStr">
					<xsl:with-param name="Language" select="$Language"/>
					<xsl:with-param name="TitleName" select="'ResultDel'"/>
				</xsl:call-template>
			</xsl:when>
			<!-- 強制メモリ受信ボックスからのダウンロード(SEND) -->
			<xsl:when test="$Result1 = 16">
				<xsl:call-template name="ChangeLangStr">
					<xsl:with-param name="Language" select="$Language"/>
					<xsl:with-param name="TitleName" select="'ResultSend'"/>
				</xsl:call-template>
			</xsl:when>
			<!-- 受信長オーバー -->
			<xsl:when test="$Result1 = 33">
				<xsl:call-template name="ChangeLangStr">
					<xsl:with-param name="Language" select="$Language"/>
					<xsl:with-param name="TitleName" select="'ResultLovr'"/>
				</xsl:call-template>
			</xsl:when>
			<!-- 受信頁オーバー -->
			<xsl:when test="$Result1 = 34">
				<xsl:call-template name="ChangeLangStr">
					<xsl:with-param name="Language" select="$Language"/>
					<xsl:with-param name="TitleName" select="'ResultPovr'"/>
				</xsl:call-template>
			</xsl:when>
			<!-- ファイル・エラー -->
			<xsl:when test="$Result1 = 35">
				<xsl:call-template name="ChangeLangStr">
					<xsl:with-param name="Language" select="$Language"/>
					<xsl:with-param name="TitleName" select="'ResultFil'"/>
				</xsl:call-template>
			</xsl:when>
			<!-- デコード・エラー -->
			<xsl:when test="$Result1 = 36">
				<xsl:call-template name="ChangeLangStr">
					<xsl:with-param name="Language" select="$Language"/>
					<xsl:with-param name="TitleName" select="'ResultDecErr'"/>
				</xsl:call-template>
			</xsl:when>
			<!-- ＭＤＮエラー -->
			<xsl:when test="$Result1 = 37">
				<xsl:call-template name="ChangeLangStr">
					<xsl:with-param name="Language" select="$Language"/>
					<xsl:with-param name="TitleName" select="'ResultMDN'"/>
				</xsl:call-template>
			</xsl:when>
			<!-- ＤＳＮエラー -->
			<xsl:when test="$Result1 = 38">
				<xsl:call-template name="ChangeLangStr">
					<xsl:with-param name="Language" select="$Language"/>
					<xsl:with-param name="TitleName" select="'ResultDSN'"/>
				</xsl:call-template>
			</xsl:when>
			<xsl:otherwise>
				<xsl:value-of select="'---'"/>
			</xsl:otherwise>
			</xsl:choose>
		</xsl:otherwise>
		</xsl:choose>
	</xsl:template>

	<!--     brief: 備考文字列設定
		 param: ループカウンタ
			文字数合計
			通信時間(時)
			通信時間(分)
			通信時間(秒)	-->
	<xsl:template name="MakeNotesStr">
		<!-- パラメータ -->
		<xsl:param name="LoopCount" select="0"/>
		<xsl:param name="NotesStrSum" select="0"/>
		<xsl:param name="DurationHour"/>
		<xsl:param name="DurationMin"/>
		<xsl:param name="DurationSec"/>

		<!-- 備考最大文字数 -->
		<xsl:variable name="NotesStrMAX" select="43"/>
		<!-- 最大データ数 -->
		<xsl:variable name="LoopMAX" select="23"/>

		<!-- 備考用データ -->
		<xsl:variable name="Fax2Exist" select="Fax2Exist"/>
		<xsl:variable name="ReserveReportLine" select="ReserveReportLine"/>
		<xsl:variable name="EtcL1" select="EtcL1"/>
		<xsl:variable name="EtcL2" select="EtcL2"/>
		<xsl:variable name="EtcSIP" select="EtcSIP"/>
		<xsl:variable name="EtcIPADR" select="EtcIPADR"/>
		<xsl:variable name="EtcI-FAX" select="EtcI-FAX"/>
		<xsl:variable name="EtcTMR" select="EtcTMR"/>
		<xsl:variable name="EtcPOL" select="EtcPOL"/>
		<xsl:variable name="EtcTRN" select="EtcTRN"/>
		<xsl:variable name="Etc2IN" select="Etc2IN"/>
		<xsl:variable name="EtcORG" select="EtcORG"/>
		<xsl:variable name="EtcFME" select="EtcFME"/>
		<xsl:variable name="EtcDPG" select="EtcDPG"/>
		<xsl:variable name="EtcMIX" select="EtcMIX"/>
		<xsl:variable name="EtcCALL" select="EtcCALL"/>
		<xsl:variable name="EtcCSRC" select="EtcCSRC"/>
		<xsl:variable name="EtcFWD" select="EtcFWD"/>
		<xsl:variable name="EtcFWDBoxNo" select="EtcFWDBoxNo"/>
		<xsl:variable name="EtcPC" select="EtcPC"/>
		<xsl:variable name="EtcBND" select="EtcBND"/>
		<xsl:variable name="EtcSP" select="EtcSP"/>
		<xsl:variable name="EtcFCODE" select="EtcFCODE"/>
		<xsl:variable name="EtcRTX" select="EtcRTX"/>
		<xsl:variable name="EtcRLY" select="EtcRLY"/>
		<xsl:variable name="EtcRLYBoxNo" select="EtcRLYBoxNo"/>
		<xsl:variable name="EtcMBX" select="EtcMBX"/>
		<xsl:variable name="EtcMBXBoxNo" select="EtcMBXBoxNo"/>
		<xsl:variable name="EtcBUL" select="EtcBUL"/>
		<xsl:variable name="EtcBULBoxNo" select="EtcBULBoxNo"/>

		<xsl:if test="$LoopCount &lt;= $LoopMAX">
			<xsl:choose>
			<xsl:when test="$LoopCount = 0">				
				<!-- L1 -->
				<xsl:choose>
				<xsl:when test="($EtcL1 = 1) and 
						($Fax2Exist = 1) and
						($EtcSIP = 0 and $EtcIPADR = 0 and $EtcI-FAX = 0) and
						not($DurationHour = 0 and $DurationMin = 0 and $DurationSec = 0) and
						($ReserveReportLine = 1)">
						<xsl:variable name="NotesStrTmpSum" select="$NotesStrSum+string-length(' L1')"/>
						<xsl:choose>
						<xsl:when test="$NotesStrTmpSum &lt;= $NotesStrMAX">
							<xsl:value-of select="' L1'"/>
							<xsl:call-template name="MakeNotesStr">
								<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
								<xsl:with-param name="NotesStrSum" select="$NotesStrTmpSum"/>
								<xsl:with-param name="DurationHour" select="$DurationHour"/>
								<xsl:with-param name="DurationMin" select="$DurationMin"/>
								<xsl:with-param name="DurationSec" select="$DurationSec"/>
							</xsl:call-template>
						</xsl:when>
						<xsl:otherwise>
							<xsl:call-template name="MakeNotesStr">
								<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
								<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
								<xsl:with-param name="DurationHour" select="$DurationHour"/>
								<xsl:with-param name="DurationMin" select="$DurationMin"/>
								<xsl:with-param name="DurationSec" select="$DurationSec"/>
							</xsl:call-template>
						</xsl:otherwise>
						</xsl:choose>
				</xsl:when>
				<xsl:otherwise>
					<xsl:call-template name="MakeNotesStr">
						<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
						<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
						<xsl:with-param name="DurationHour" select="$DurationHour"/>
						<xsl:with-param name="DurationMin" select="$DurationMin"/>
						<xsl:with-param name="DurationSec" select="$DurationSec"/>
					</xsl:call-template>
				</xsl:otherwise>
				</xsl:choose>
			</xsl:when>
			<xsl:when test="$LoopCount = 1">
				<!-- L2 -->
				<xsl:choose>
				<xsl:when test="($EtcL2 = 1) and
						($Fax2Exist = 1) and
						($EtcSIP = 0 and $EtcIPADR = 0 and $EtcI-FAX = 0) and
						not($DurationHour = 0 and $DurationMin = 0 and $DurationSec = 0) and
						($ReserveReportLine = 1)">
						<xsl:variable name="NotesStrTmpSum" select="$NotesStrSum+string-length(' L2')"/>
						<xsl:choose>
						<xsl:when test="$NotesStrTmpSum &lt;= $NotesStrMAX">
							<xsl:value-of select="' L2'"/>
							<xsl:call-template name="MakeNotesStr">
								<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
								<xsl:with-param name="NotesStrSum" select="$NotesStrTmpSum"/>
							</xsl:call-template>
						</xsl:when>
						<xsl:otherwise>
							<xsl:call-template name="MakeNotesStr">
								<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
								<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
							</xsl:call-template>
						</xsl:otherwise>
						</xsl:choose>

				</xsl:when>
				<xsl:otherwise>
					<xsl:call-template name="MakeNotesStr">
						<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
						<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
					</xsl:call-template>
				</xsl:otherwise>
				</xsl:choose>
			</xsl:when>
			<!-- SIP -->
			<xsl:when test="$LoopCount = 2">
				<xsl:choose>
				<xsl:when test="$EtcSIP = 1">
					<xsl:variable name="NotesStrTmpSum" select="$NotesStrSum+string-length(' SIP')"/>
					<xsl:choose>
					<xsl:when test="$NotesStrTmpSum &lt;= $NotesStrMAX">
						<xsl:value-of select="' SIP'"/>
						<xsl:call-template name="MakeNotesStr">
							<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
							<xsl:with-param name="NotesStrSum" select="$NotesStrTmpSum"/>
						</xsl:call-template>
					</xsl:when>
					<xsl:otherwise>
						<xsl:call-template name="MakeNotesStr">
							<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
							<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
						</xsl:call-template>
					</xsl:otherwise>
					</xsl:choose>
				</xsl:when>
				<xsl:otherwise>
					<xsl:call-template name="MakeNotesStr">
						<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
						<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
					</xsl:call-template>
				</xsl:otherwise>
				</xsl:choose>
			</xsl:when>
			<!-- IPADR -->
			<xsl:when test="$LoopCount = 3">
				<xsl:choose>
				<xsl:when test="$EtcIPADR = 1">
					<xsl:variable name="NotesStrTmpSum" select="$NotesStrSum+string-length(' IPADR')"/>
					<xsl:choose>
					<xsl:when test="$NotesStrTmpSum &lt;= $NotesStrMAX">
						<xsl:value-of select="' IPADR'"/>
						<xsl:call-template name="MakeNotesStr">
							<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
							<xsl:with-param name="NotesStrSum" select="$NotesStrTmpSum"/>
						</xsl:call-template>
					</xsl:when>
					<xsl:otherwise>
						<xsl:call-template name="MakeNotesStr">
							<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
							<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
						</xsl:call-template>
					</xsl:otherwise>
					</xsl:choose>
				</xsl:when>
				<xsl:otherwise>
					<xsl:call-template name="MakeNotesStr">
						<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
						<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
					</xsl:call-template>
				</xsl:otherwise>
				</xsl:choose>
			</xsl:when>
			<!-- I-FAX -->
			<xsl:when test="$LoopCount = 4">
				<xsl:choose>
				<xsl:when test="$EtcI-FAX = 1">
					<xsl:variable name="NotesStrTmpSum" select="$NotesStrSum+string-length(' I-FAX')"/>
					<xsl:choose>
					<xsl:when test="$NotesStrTmpSum &lt;= $NotesStrMAX">
						<xsl:value-of select="' I-FAX'"/>
						<xsl:call-template name="MakeNotesStr">
							<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
							<xsl:with-param name="NotesStrSum" select="$NotesStrTmpSum"/>
						</xsl:call-template>
					</xsl:when>
					<xsl:otherwise>
						<xsl:call-template name="MakeNotesStr">
							<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
							<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
						</xsl:call-template>
					</xsl:otherwise>
					</xsl:choose>
				</xsl:when>
				<xsl:otherwise>
					<xsl:call-template name="MakeNotesStr">
						<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
						<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
					</xsl:call-template>
				</xsl:otherwise>
				</xsl:choose>
			</xsl:when>
			<!-- TMR -->
			<xsl:when test="$LoopCount = 5">
				<xsl:choose>
				<xsl:when test="$EtcTMR = 1">
					<xsl:variable name="NotesStrTmpSum" select="$NotesStrSum+string-length(' TMR')"/>
					<xsl:choose>
					<xsl:when test="$NotesStrTmpSum &lt;= $NotesStrMAX">
						<xsl:value-of select="' TMR'"/>
						<xsl:call-template name="MakeNotesStr">
							<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
							<xsl:with-param name="NotesStrSum" select="$NotesStrTmpSum"/>
						</xsl:call-template>
					</xsl:when>
					<xsl:otherwise>
						<xsl:call-template name="MakeNotesStr">
							<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
							<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
						</xsl:call-template>
					</xsl:otherwise>
					</xsl:choose>
				</xsl:when>
				<xsl:otherwise>
					<xsl:call-template name="MakeNotesStr">
						<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
						<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
					</xsl:call-template>
				</xsl:otherwise>
				</xsl:choose>
			</xsl:when>
			<!-- POL -->
			<xsl:when test="$LoopCount = 6">
				<xsl:choose>
				<xsl:when test="$EtcPOL = 1">
					<xsl:variable name="NotesStrTmpSum" select="$NotesStrSum+string-length(' POL')"/>
					<xsl:choose>
					<xsl:when test="$NotesStrTmpSum &lt;= $NotesStrMAX">
						<xsl:value-of select="' POL'"/>
						<xsl:call-template name="MakeNotesStr">
							<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
							<xsl:with-param name="NotesStrSum" select="$NotesStrTmpSum"/>
						</xsl:call-template>
					</xsl:when>
					<xsl:otherwise>
						<xsl:call-template name="MakeNotesStr">
							<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
							<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
						</xsl:call-template>
					</xsl:otherwise>
					</xsl:choose>
				</xsl:when>
				<xsl:otherwise>
					<xsl:call-template name="MakeNotesStr">
						<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
						<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
					</xsl:call-template>
				</xsl:otherwise>
				</xsl:choose>
			</xsl:when>
			<!-- TRN -->
			<xsl:when test="$LoopCount = 7">
				<xsl:choose>
				<xsl:when test="$EtcTRN = 1">
					<xsl:variable name="NotesStrTmpSum" select="$NotesStrSum+string-length(' TRN')"/>
					<xsl:choose>
					<xsl:when test="$NotesStrTmpSum &lt;= $NotesStrMAX">
						<xsl:value-of select="' TRN'"/>
						<xsl:call-template name="MakeNotesStr">
							<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
							<xsl:with-param name="NotesStrSum" select="$NotesStrTmpSum"/>
						</xsl:call-template>
					</xsl:when>
					<xsl:otherwise>
						<xsl:call-template name="MakeNotesStr">
							<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
							<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
						</xsl:call-template>
					</xsl:otherwise>
					</xsl:choose>
				</xsl:when>
				<xsl:otherwise>
					<xsl:call-template name="MakeNotesStr">
						<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
						<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
					</xsl:call-template>
				</xsl:otherwise>
				</xsl:choose>
			</xsl:when>
			<!-- 2IN -->
			<xsl:when test="$LoopCount = 8">
				<xsl:choose>
				<xsl:when test="$Etc2IN = 1">
					<xsl:variable name="NotesStrTmpSum" select="$NotesStrSum+string-length(' 2IN')"/>
					<xsl:choose>
					<xsl:when test="$NotesStrTmpSum &lt;= $NotesStrMAX">
						<xsl:value-of select="' 2IN'"/>
						<xsl:call-template name="MakeNotesStr">
							<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
							<xsl:with-param name="NotesStrSum" select="$NotesStrTmpSum"/>
						</xsl:call-template>
					</xsl:when>
					<xsl:otherwise>
						<xsl:call-template name="MakeNotesStr">
							<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
							<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
						</xsl:call-template>
					</xsl:otherwise>
					</xsl:choose>
				</xsl:when>
				<xsl:otherwise>
					<xsl:call-template name="MakeNotesStr">
						<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
						<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
					</xsl:call-template>
				</xsl:otherwise>
				</xsl:choose>
			</xsl:when>
			<!-- ORG -->
			<xsl:when test="$LoopCount = 9">
				<xsl:choose>
				<xsl:when test="$EtcORG = 1">
					<xsl:variable name="NotesStrTmpSum" select="$NotesStrSum+string-length(' ORG')"/>
					<xsl:choose>
					<xsl:when test="$NotesStrTmpSum &lt;= $NotesStrMAX">
						<xsl:value-of select="' ORG'"/>
						<xsl:call-template name="MakeNotesStr">
							<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
							<xsl:with-param name="NotesStrSum" select="$NotesStrTmpSum"/>
						</xsl:call-template>
					</xsl:when>
					<xsl:otherwise>
						<xsl:call-template name="MakeNotesStr">
							<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
							<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
						</xsl:call-template>
					</xsl:otherwise>
					</xsl:choose>
				</xsl:when>
				<xsl:otherwise>
					<xsl:call-template name="MakeNotesStr">
						<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
						<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
					</xsl:call-template>
				</xsl:otherwise>
				</xsl:choose>
			</xsl:when>
			<!-- FME -->
			<xsl:when test="$LoopCount = 10">
				<xsl:choose>
				<xsl:when test="$EtcFME = 1">
					<xsl:variable name="NotesStrTmpSum" select="$NotesStrSum+string-length(' FME')"/>
					<xsl:choose>
					<xsl:when test="$NotesStrTmpSum &lt;= $NotesStrMAX">
						<xsl:value-of select="' FME'"/>
						<xsl:call-template name="MakeNotesStr">
							<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
							<xsl:with-param name="NotesStrSum" select="$NotesStrTmpSum"/>
						</xsl:call-template>
					</xsl:when>
					<xsl:otherwise>
						<xsl:call-template name="MakeNotesStr">
							<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
							<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
						</xsl:call-template>
					</xsl:otherwise>
					</xsl:choose>
				</xsl:when>
				<xsl:otherwise>
					<xsl:call-template name="MakeNotesStr">
						<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
						<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
					</xsl:call-template>
				</xsl:otherwise>
				</xsl:choose>
			</xsl:when>
			<!-- DPG -->
			<xsl:when test="$LoopCount = 11">
				<xsl:choose>
				<xsl:when test="$EtcDPG = 1">
					<xsl:variable name="NotesStrTmpSum" select="$NotesStrSum+string-length(' DPG')"/>
					<xsl:choose>
					<xsl:when test="$NotesStrTmpSum &lt;= $NotesStrMAX">
						<xsl:value-of select="' DPG'"/>
						<xsl:call-template name="MakeNotesStr">
							<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
							<xsl:with-param name="NotesStrSum" select="$NotesStrTmpSum"/>
						</xsl:call-template>
					</xsl:when>
					<xsl:otherwise>
						<xsl:call-template name="MakeNotesStr">
							<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
							<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
						</xsl:call-template>
					</xsl:otherwise>
					</xsl:choose>
				</xsl:when>
				<xsl:otherwise>
					<xsl:call-template name="MakeNotesStr">
						<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
						<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
					</xsl:call-template>
				</xsl:otherwise>
				</xsl:choose>
			</xsl:when>
			<!-- MIX -->
			<xsl:when test="$LoopCount = 12">
				<xsl:choose>
				<xsl:when test="$EtcMIX = 1">
					<xsl:variable name="NotesStrTmpSum" select="$NotesStrSum+string-length(' MIX')"/>
					<xsl:choose>
					<xsl:when test="$NotesStrTmpSum &lt;= $NotesStrMAX">
						<xsl:value-of select="' MIX'"/>
						<xsl:call-template name="MakeNotesStr">
							<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
							<xsl:with-param name="NotesStrSum" select="$NotesStrTmpSum"/>
						</xsl:call-template>
					</xsl:when>
					<xsl:otherwise>
						<xsl:call-template name="MakeNotesStr">
							<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
							<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
						</xsl:call-template>
					</xsl:otherwise>
					</xsl:choose>
				</xsl:when>
				<xsl:otherwise>
					<xsl:call-template name="MakeNotesStr">
						<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
						<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
					</xsl:call-template>
				</xsl:otherwise>
				</xsl:choose>
			</xsl:when>
			<!-- CALL -->
			<xsl:when test="$LoopCount = 13">
				<xsl:choose>
				<xsl:when test="$EtcCALL = 1">
					<xsl:variable name="NotesStrTmpSum" select="$NotesStrSum+string-length(' CALL')"/>
					<xsl:choose>
					<xsl:when test="$NotesStrTmpSum &lt;= $NotesStrMAX">
						<xsl:value-of select="' CALL'"/>
						<xsl:call-template name="MakeNotesStr">
							<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
							<xsl:with-param name="NotesStrSum" select="$NotesStrTmpSum"/>
						</xsl:call-template>
					</xsl:when>
					<xsl:otherwise>
						<xsl:call-template name="MakeNotesStr">
							<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
							<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
						</xsl:call-template>
					</xsl:otherwise>
					</xsl:choose>
				</xsl:when>
				<xsl:otherwise>
					<xsl:call-template name="MakeNotesStr">
						<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
						<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
					</xsl:call-template>
				</xsl:otherwise>
				</xsl:choose>
			</xsl:when>
			<!-- CSRC -->
			<xsl:when test="$LoopCount = 14">
				<xsl:choose>
				<xsl:when test="$EtcCSRC = 1">
					<xsl:variable name="NotesStrTmpSum" select="$NotesStrSum+string-length(' CSRC')"/>
					<xsl:choose>
					<xsl:when test="$NotesStrTmpSum &lt;= $NotesStrMAX">
						<xsl:value-of select="' CSRC'"/>
						<xsl:call-template name="MakeNotesStr">
							<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
							<xsl:with-param name="NotesStrSum" select="$NotesStrTmpSum"/>
						</xsl:call-template>
					</xsl:when>
					<xsl:otherwise>
						<xsl:call-template name="MakeNotesStr">
							<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
							<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
						</xsl:call-template>
					</xsl:otherwise>
					</xsl:choose>
				</xsl:when>
				<xsl:otherwise>
					<xsl:call-template name="MakeNotesStr">
						<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
						<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
					</xsl:call-template>
				</xsl:otherwise>
				</xsl:choose>
			</xsl:when>
			<!-- FWD -->
			<xsl:when test="$LoopCount = 15">
				<xsl:choose>
				<xsl:when test="$EtcFWD = 1">
					<xsl:variable name="NotesStrTmpSum" select="$NotesStrSum+string-length(' FWD')+9"/>
					<xsl:choose>
					<xsl:when test="$NotesStrTmpSum &lt;= $NotesStrMAX">
						<xsl:value-of select="' FWD'"/>
						<xsl:value-of select="EtcFWDBoxNo"/>
						<xsl:call-template name="MakeNotesStr">
							<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
							<xsl:with-param name="NotesStrSum" select="$NotesStrTmpSum"/>
						</xsl:call-template>
					</xsl:when>
					<xsl:otherwise>
						<xsl:call-template name="MakeNotesStr">
							<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
							<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
						</xsl:call-template>
					</xsl:otherwise>
					</xsl:choose>
				</xsl:when>
				<xsl:otherwise>
					<xsl:call-template name="MakeNotesStr">
						<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
						<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
					</xsl:call-template>
				</xsl:otherwise>
				</xsl:choose>
			</xsl:when>
			<!-- PC -->
			<xsl:when test="$LoopCount = 16">
				<xsl:choose>
				<xsl:when test="$EtcPC = 1">
					<xsl:variable name="NotesStrTmpSum" select="$NotesStrSum+string-length(' PC')"/>
					<xsl:choose>
					<xsl:when test="$NotesStrTmpSum &lt;= $NotesStrMAX">
						<xsl:value-of select="' PC'"/>
						<xsl:call-template name="MakeNotesStr">
							<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
							<xsl:with-param name="NotesStrSum" select="$NotesStrTmpSum"/>
						</xsl:call-template>
					</xsl:when>
					<xsl:otherwise>
						<xsl:call-template name="MakeNotesStr">
							<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
							<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
						</xsl:call-template>
					</xsl:otherwise>
					</xsl:choose>
				</xsl:when>
				<xsl:otherwise>
					<xsl:call-template name="MakeNotesStr">
						<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
						<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
					</xsl:call-template>
				</xsl:otherwise>
				</xsl:choose>
			</xsl:when>
			<!-- BND -->
			<xsl:when test="$LoopCount = 17">
				<xsl:choose>
				<xsl:when test="$EtcBND = 1">
					<xsl:variable name="NotesStrTmpSum" select="$NotesStrSum+string-length(' BND')"/>
					<xsl:choose>
					<xsl:when test="$NotesStrTmpSum &lt;= $NotesStrMAX">
						<xsl:value-of select="' BND'"/>
						<xsl:call-template name="MakeNotesStr">
							<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
							<xsl:with-param name="NotesStrSum" select="$NotesStrTmpSum"/>
						</xsl:call-template>
					</xsl:when>
					<xsl:otherwise>
						<xsl:call-template name="MakeNotesStr">
							<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
							<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
						</xsl:call-template>
					</xsl:otherwise>
					</xsl:choose>
				</xsl:when>
				<xsl:otherwise>
					<xsl:call-template name="MakeNotesStr">
						<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
						<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
					</xsl:call-template>
				</xsl:otherwise>
				</xsl:choose>
			</xsl:when>
			<!-- SP -->
			<xsl:when test="$LoopCount = 18">
				<xsl:choose>
				<xsl:when test="$EtcSP = 1">
					<xsl:variable name="NotesStrTmpSum" select="$NotesStrSum+string-length(' SP')"/>
					<xsl:choose>
					<xsl:when test="$NotesStrTmpSum &lt;= $NotesStrMAX">
						<xsl:value-of select="' SP'"/>
						<xsl:call-template name="MakeNotesStr">
							<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
							<xsl:with-param name="NotesStrSum" select="$NotesStrTmpSum"/>
						</xsl:call-template>
					</xsl:when>
					<xsl:otherwise>
						<xsl:call-template name="MakeNotesStr">
							<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
							<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
						</xsl:call-template>
					</xsl:otherwise>
					</xsl:choose>
				</xsl:when>
				<xsl:otherwise>
					<xsl:call-template name="MakeNotesStr">
						<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
						<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
					</xsl:call-template>
				</xsl:otherwise>
				</xsl:choose>
			</xsl:when>
			<!-- FCODE -->
			<xsl:when test="$LoopCount = 19">
				<xsl:choose>
				<xsl:when test="$EtcFCODE = 1">
					<xsl:variable name="NotesStrTmpSum" select="$NotesStrSum+string-length(' FCODE')"/>
					<xsl:choose>
					<xsl:when test="$NotesStrTmpSum &lt;= $NotesStrMAX">
						<xsl:value-of select="' FCODE'"/>
						<xsl:call-template name="MakeNotesStr">
							<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
							<xsl:with-param name="NotesStrSum" select="$NotesStrTmpSum"/>
						</xsl:call-template>
					</xsl:when>
					<xsl:otherwise>
						<xsl:call-template name="MakeNotesStr">
							<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
							<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
						</xsl:call-template>
					</xsl:otherwise>
					</xsl:choose>
				</xsl:when>
				<xsl:otherwise>
					<xsl:call-template name="MakeNotesStr">
						<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
						<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
					</xsl:call-template>
				</xsl:otherwise>
				</xsl:choose>
			</xsl:when>
			<!-- RTX -->
			<xsl:when test="$LoopCount = 20">
				<xsl:choose>
				<xsl:when test="$EtcRTX = 1">
					<xsl:variable name="NotesStrTmpSum" select="$NotesStrSum+string-length(' RTX')"/>
					<xsl:choose>
					<xsl:when test="$NotesStrTmpSum &lt;= $NotesStrMAX">
						<xsl:value-of select="' RTX'"/>
						<xsl:call-template name="MakeNotesStr">
							<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
							<xsl:with-param name="NotesStrSum" select="$NotesStrTmpSum"/>
						</xsl:call-template>
					</xsl:when>
					<xsl:otherwise>
						<xsl:call-template name="MakeNotesStr">
							<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
							<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
						</xsl:call-template>
					</xsl:otherwise>
					</xsl:choose>
				</xsl:when>
				<xsl:otherwise>
					<xsl:call-template name="MakeNotesStr">
						<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
						<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
					</xsl:call-template>
				</xsl:otherwise>
				</xsl:choose>
			</xsl:when>
			<!-- RLY -->
			<xsl:when test="$LoopCount = 21">
				<xsl:choose>
				<xsl:when test="$EtcRLY = 1">
					<xsl:variable name="NotesStrTmpSum" select="$NotesStrSum+string-length(' RLY')+9"/>
					<xsl:choose>
					<xsl:when test="$NotesStrTmpSum &lt;= $NotesStrMAX">
						<xsl:value-of select="' RLY'"/>
						<xsl:value-of select="EtcRLYBoxNo"/>
						<xsl:call-template name="MakeNotesStr">
							<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
							<xsl:with-param name="NotesStrSum" select="$NotesStrTmpSum"/>
						</xsl:call-template>
					</xsl:when>
					<xsl:otherwise>
						<xsl:call-template name="MakeNotesStr">
							<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
							<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
						</xsl:call-template>
					</xsl:otherwise>
					</xsl:choose>
				</xsl:when>
				<xsl:otherwise>
					<xsl:call-template name="MakeNotesStr">
						<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
						<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
					</xsl:call-template>
				</xsl:otherwise>
				</xsl:choose>
			</xsl:when>
			<!-- MBX -->
			<xsl:when test="$LoopCount = 22">
				<xsl:choose>
				<xsl:when test="$EtcMBX = 1">
					<xsl:variable name="NotesStrTmpSum" select="$NotesStrSum+string-length(' MBX')+9"/>
					<xsl:choose>
					<xsl:when test="$NotesStrTmpSum &lt;= $NotesStrMAX">
						<xsl:value-of select="' MBX'"/>
						<xsl:value-of select="EtcMBXBoxNo"/>
						<xsl:call-template name="MakeNotesStr">
							<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
							<xsl:with-param name="NotesStrSum" select="$NotesStrTmpSum"/>
						</xsl:call-template>
					</xsl:when>
					<xsl:otherwise>
						<xsl:call-template name="MakeNotesStr">
							<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
							<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
						</xsl:call-template>
					</xsl:otherwise>
					</xsl:choose>
				</xsl:when>
				<xsl:otherwise>
					<xsl:call-template name="MakeNotesStr">
						<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
						<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
					</xsl:call-template>
				</xsl:otherwise>
				</xsl:choose>
			</xsl:when>
			<!-- BUL -->
			<xsl:when test="$LoopCount = 23">
				<xsl:choose>
				<xsl:when test="$EtcBUL = 1">
					<xsl:variable name="NotesStrTmpSum" select="$NotesStrSum+string-length(' BUL')+9"/>
					<xsl:choose>
					<xsl:when test="$NotesStrTmpSum &lt;= $NotesStrMAX">
						<xsl:value-of select="' BUL'"/>
						<xsl:value-of select="EtcBULBoxNo"/>
						<xsl:call-template name="MakeNotesStr">
							<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
							<xsl:with-param name="NotesStrSum" select="$NotesStrTmpSum"/>
						</xsl:call-template>
					</xsl:when>
					<xsl:otherwise>
						<xsl:call-template name="MakeNotesStr">
							<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
							<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
						</xsl:call-template>
					</xsl:otherwise>
					</xsl:choose>
				</xsl:when>
				<xsl:otherwise>
					<xsl:call-template name="MakeNotesStr">
						<xsl:with-param name="LoopCount" select="$LoopCount+1"/>
						<xsl:with-param name="NotesStrSum" select="$NotesStrSum"/>
					</xsl:call-template>
				</xsl:otherwise>
				</xsl:choose>
			</xsl:when>
			</xsl:choose>
		</xsl:if>
	</xsl:template>

	<!--    brief: マルチ言語対応
		param: 対応言語設定
		　　　 設定タイトル　　　　-->
	<xsl:template name="ChangeLangStr">
		<!-- パラメータ -->
		<xsl:param name="Language"/>
		<xsl:param name="TitleName"/>

		<xsl:choose>
			<!-- 日本語 -->
			<xsl:when test="$Language = 0">
				<xsl:if test="$TitleName = 'ReportTitle'">通信結果レポート</xsl:if>
				<xsl:if test="$TitleName = 'RemoteIDTitle'">相手先</xsl:if>
				<xsl:if test="$TitleName = 'StartTimeTitle'">開始時刻</xsl:if>
				<xsl:if test="$TitleName = 'DurationTimeTitle'">時間</xsl:if>
				<xsl:if test="$TitleName = 'PageNoTitle'">枚数</xsl:if>
				<xsl:if test="$TitleName = 'ResultTitle'">結果</xsl:if>
				<xsl:if test="$TitleName = 'NotesTitle'">備考</xsl:if>
				<xsl:if test="$TitleName = 'NotesExplanation'">
				TMR:ﾀｲﾏｰ通信 POL:ﾎﾟｰﾘﾝｸﾞ ORG:原稿サイズ指定 FME:枠消し送信 DPG:ﾍﾟｰｼﾞ連写送信<br/>
				MIX:混載ﾓｰﾄﾞ送信 CALL:手動送信 CSRC:CSRC FWD:転送 PC:PC-FAX<br/>
				BND:両面とじ方向 SP:特殊原稿 FCODE:Fｺｰﾄﾞ指定 RTX:再送信 RLY:中継 MBX:親展 BUL:提示版<br/>
				SIP:SIPﾌｧｸｽ IPADR:IPｱﾄﾞﾚｽﾌｧｸｽ I-FAX:ｲﾝﾀｰﾈｯﾄﾌｧｸｽ
				</xsl:if>
				<xsl:if test="$TitleName = 'ResultExplanation'">
				OK:通信OK終了 S-OK:通信中止 PW-OFF:電源OFF TEL:電話からの受信 NG:その他のエラー 継続:継続<br/>
				無応答:無応答 拒否:着信拒否 話中:話中 ﾒﾓﾘｰﾌﾙ:ﾒﾓﾘｰﾌﾙ LOVR:受信長ｵｰﾊﾞｰ POVR:受信ﾍﾟｰｼﾞｵｰﾊﾞｰ<br/>
				FIL:ﾌｧｲﾙｴﾗｰ DC:ﾃﾞｺｰﾄﾞｴﾗｰ MDN:MDN応答ｴﾗｰ DSN:DSN応答ｴﾗｰ<br/>
				PRINT:強制ﾒﾓﾘ文章印刷 DEL:強制ﾒﾓﾘ文章削除 SEND:強制ﾒﾓﾘ文書送信
				</xsl:if>
				<xsl:if test="$TitleName = 'ResultOK'">OK</xsl:if>
				<xsl:if test="$TitleName = 'ResultS-OK'">S-OK</xsl:if>
				<xsl:if test="$TitleName = 'ResultPW-OFF'">PW-OFF</xsl:if>
				<xsl:if test="$TitleName = 'ResultTEL'">TEL</xsl:if>
				<xsl:if test="$TitleName = 'ResultNG'">NG</xsl:if>
				<xsl:if test="$TitleName = 'ResultCont'">継続</xsl:if>
				<xsl:if test="$TitleName = 'ResultNoAns'">無応答</xsl:if>
				<xsl:if test="$TitleName = 'ResultRefuse'">拒否</xsl:if>
				<xsl:if test="$TitleName = 'ResultBusy'">話中</xsl:if>
				<xsl:if test="$TitleName = 'ResultM-Full'">ﾒﾓﾘｰﾌﾙ</xsl:if>
				<xsl:if test="$TitleName = 'ResultScanJamErr'">J00</xsl:if>
				<xsl:if test="$TitleName = 'ResultPrint'">PRINT</xsl:if>
				<xsl:if test="$TitleName = 'ResultDel'">DEL</xsl:if>
				<xsl:if test="$TitleName = 'ResultSend'">SEND</xsl:if>
				<xsl:if test="$TitleName = 'ResultLovr'">LOVR</xsl:if>
				<xsl:if test="$TitleName = 'ResultPovr'">POVR</xsl:if>
				<xsl:if test="$TitleName = 'ResultFil'">FIL</xsl:if>
				<xsl:if test="$TitleName = 'ResultDecErr'">DC</xsl:if>
				<xsl:if test="$TitleName = 'ResultMDN'">MDN</xsl:if>
				<xsl:if test="$TitleName = 'ResultDSN'">DSN</xsl:if>
			</xsl:when>
			<!-- 英語 
			<xsl:when test="$Language = 1"></xsl:when>
			-->
			<!-- ドイツ語 
			<xsl:when test="$Language = 2"></xsl:when>
			-->
			<!-- フランス語 
			<xsl:when test="$Language = 3"></xsl:when>
			-->
			<!-- イタリア語
			<xsl:when test="$Language = 4"></xsl:when>
			-->
			<!-- デンマーク語
			<xsl:when test="$Language = 5"></xsl:when>
			-->
			<!-- オランダ゛語 
			<xsl:when test="$Language = 6"></xsl:when>
			-->
			<!-- ポルトガル語 
			<xsl:when test="$Language = 7"></xsl:when>
			-->
			<!-- スウェーデン語 
			<xsl:when test="$Language = 8"></xsl:when>
			-->
			<!-- スペイン語 
			<xsl:when test="$Language = 9"></xsl:when>
			-->
			<!-- ノルウェー語 
			<xsl:when test="$Language = 10"></xsl:when>
			-->
			<!-- フィンランド語 
			<xsl:when test="$Language = 11"></xsl:when>
			-->
			<!-- チェコ語 
			<xsl:when test="$Language = 12"></xsl:when>
			-->
			<!-- ハンガリー語 
			<xsl:when test="$Language = 13"></xsl:when>
			-->
			<!-- ギリシア語 
			<xsl:when test="$Language = 14"></xsl:when>
			-->
			<!-- トルコ語 
			<xsl:when test="$Language = 15"></xsl:when>
			-->
			<!-- 広東語 
			<xsl:when test="$Language = 16"></xsl:when>
			-->
			<!-- 中国語 
			<xsl:when test="$Language = 17"></xsl:when>
			-->
			<!-- ハングル語
			<xsl:when test="$Language = 18"></xsl:when>
			 -->
			<!-- ロシア語 
			<xsl:when test="$Language = 19"></xsl:when>
			-->
			<!-- クロアチア語 
			<xsl:when test="$Language = 20"></xsl:when>
			-->
			<!-- ウクライナ語 
			<xsl:when test="$Language = 21"></xsl:when>
			-->
			<!-- ルーマニア語 
			<xsl:when test="$Language = 22"></xsl:when>
			-->
			<!-- スロベキア語 
			<xsl:when test="$Language = 23"></xsl:when>
			-->
			<!-- アラビア語 
			<xsl:when test="$Language = 24"></xsl:when>
			-->
			<!-- タイ語 
			<xsl:when test="$Language = 25"></xsl:when>
			-->
			<!-- マレーシア語 
			<xsl:when test="$Language = 26"></xsl:when>
			-->
			<!-- ポーランド語 
			<xsl:when test="$Language = 27"></xsl:when>
			-->
			
			<!-- 日本向け/default -->
			<xsl:otherwise>
				<xsl:if test="$TitleName = 'ReportTitle'">通信結果レポート</xsl:if>
				<xsl:if test="$TitleName = 'RemoteIDTitle'">相手先</xsl:if>
				<xsl:if test="$TitleName = 'StartTimeTitle'">開始時刻</xsl:if>
				<xsl:if test="$TitleName = 'DurationTimeTitle'">時間</xsl:if>
				<xsl:if test="$TitleName = 'PageNoTitle'">枚数</xsl:if>
				<xsl:if test="$TitleName = 'ResultTitle'">結果</xsl:if>
				<xsl:if test="$TitleName = 'NotesTitle'">備考</xsl:if>
				<xsl:if test="$TitleName = 'NotesExplanation'">
				TMR:ﾀｲﾏｰ通信 POL:ﾎﾟｰﾘﾝｸﾞ ORG:原稿サイズ指定 FME:枠消し送信 DPG:ﾍﾟｰｼﾞ連写送信<br/>
				MIX:混載ﾓｰﾄﾞ送信 CALL:手動送信 CSRC:CSRC FWD:転送 PC:PC-FAX<br/>
				BND:両面とじ方向 SP:特殊原稿 FCODE:Fｺｰﾄﾞ指定 RTX:再送信 RLY:中継 MBX:親展 BUL:提示版<br/>
				SIP:SIPﾌｧｸｽ IPADR:IPｱﾄﾞﾚｽﾌｧｸｽ I-FAX:ｲﾝﾀｰﾈｯﾄﾌｧｸｽ
				</xsl:if>
				<xsl:if test="$TitleName = 'ResultExplanation'">
				OK:通信OK終了 S-OK:通信中止 PW-OFF:電源OFF TEL:電話からの受信 NG:その他のエラー 継続:継続<br/>
				無応答:無応答 拒否:着信拒否 話中:話中 ﾒﾓﾘｰﾌﾙ:ﾒﾓﾘｰﾌﾙ LOVR:受信長ｵｰﾊﾞｰ POVR:受信ﾍﾟｰｼﾞｵｰﾊﾞｰ<br/>
				FIL:ﾌｧｲﾙｴﾗｰ DC:ﾃﾞｺｰﾄﾞｴﾗｰ MDN:MDN応答ｴﾗｰ DSN:DSN応答ｴﾗｰ<br/>
				PRINT:強制ﾒﾓﾘ文章印刷 DEL:強制ﾒﾓﾘ文章削除 SEND:強制ﾒﾓﾘ文書送信
				</xsl:if>
				<xsl:if test="$TitleName = 'ResultOK'">OK</xsl:if>
				<xsl:if test="$TitleName = 'ResultS-OK'">S-OK</xsl:if>
				<xsl:if test="$TitleName = 'ResultPW-OFF'">PW-OFF</xsl:if>
				<xsl:if test="$TitleName = 'ResultTEL'">TEL</xsl:if>
				<xsl:if test="$TitleName = 'ResultNG'">NG</xsl:if>
				<xsl:if test="$TitleName = 'ResultCont'">継続</xsl:if>
				<xsl:if test="$TitleName = 'ResultNoAns'">無応答</xsl:if>
				<xsl:if test="$TitleName = 'ResultRefuse'">拒否</xsl:if>
				<xsl:if test="$TitleName = 'ResultBusy'">話中</xsl:if>
				<xsl:if test="$TitleName = 'ResultM-Full'">ﾒﾓﾘｰﾌﾙ</xsl:if>
				<xsl:if test="$TitleName = 'ResultScanJamErr'">J00</xsl:if>
				<xsl:if test="$TitleName = 'ResultPrint'">PRINT</xsl:if>
				<xsl:if test="$TitleName = 'ResultDel'">DEL</xsl:if>
				<xsl:if test="$TitleName = 'ResultSend'">SEND</xsl:if>
				<xsl:if test="$TitleName = 'ResultLovr'">LOVR</xsl:if>
				<xsl:if test="$TitleName = 'ResultPovr'">POVR</xsl:if>
				<xsl:if test="$TitleName = 'ResultFil'">FIL</xsl:if>
				<xsl:if test="$TitleName = 'ResultDecErr'">DC</xsl:if>
				<xsl:if test="$TitleName = 'ResultMDN'">MDN</xsl:if>
				<xsl:if test="$TitleName = 'ResultDSN'">DSN</xsl:if>
			</xsl:otherwise>
		</xsl:choose>
	</xsl:template>

</xsl:stylesheet>